/* FUNCTION:     Numerically solve an initial value problem.
   AUTHORS:      Lawrence Shampine, Richard Allen, Steven Pruess  for 
                 the text  Fundamentals of Numerical Computing
   DATE:         December 9, 1995   
   LAST CHANGE:  July 3, 1996                            */

#include "fnc.h"
#include <float.h>       /*  for DBL_EPSILON   */
#include <math.h>        /*  for pow  */
#include <stdlib.h>      /*  for malloc  */
#include <stdio.h>       /*  for NULL   */

void England();
void Yvalue();

void Rke(f, neq, x, y, h, first, tol, threshold, flag, step, ycoeff)
     void (*f)( );
     int neq, *first, *flag;
     double *x, y[], *h, tol, threshold[], *step, ycoeff[];
{
/*
  Function Rke integrates a system of neq first order ordinary differ-
  ential equations over one step using a Runge-Kutta method due to R.
  England.  It provides for the evaluation of the solution within the
  step by interpolation.  The England formulas are evaluated in
  function England and the quintic Hermite interpolant is evaluated in
  the function Yvalue.  The function Yvalue can be used independently
  after any successful step by Rke to approximate the solution
  components in the interval [x-step,x].

  Input parameters:
   f     = name of the function defining the system of differential
           equations.  This function must have the form:
                void f(x, y, yprime)
                double x, y[], yprime[];
                {
                   yprime[0] = ...;
                       ...
                   yprime[neq-1] = ...;
                }
                return;
   neq   = number of first order equations in the system.
   x     = initial value of the independent variable.
   y     = vector (with length neq) of solution values at x.
   h     = step size for current step (its sign determines the
           direction of integration).  On the first call to Rke you
           must specify h.  After the first call, the code suggests
           a suitable h for the next step.
   first = "logical" variable indicating first call or subsequent call.
           On the first call to Rke set first = true (nonzero); for
           subsequent calls the code automatically sets first = false.
   tol, threshold = desired tolerances: tol is a scalar and threshold
           is a vector with neq components.  The value of tol must be
           in the interval [ 10*u , 0.01 ] where u is the unit roundoff
           for your machine; each component of threshold must be
           non-negative.  On the first call, if some y[i] = 0 then
           threshold[i] must be positive for that i.  The convergence
           criterion is roughly
                abs(local error in y[i]) <=
                      tol*max(threshold[i],abs(y[i]))
           for each i.

  Output parameters:
   x      = value of the independent variable to which the
            integration advanced.
   y      = vector of computed solution values at x.
   h      = step size suitable for the next step.
   flag   = an integer reporting what the code did:
              0, if a step was successfully taken;
              1, if excessive work (more than 500 steps) was expended;
              2, if tol, threshold are too small;
             -1, if input is invalid;
             -2, if memory allocation failed.
   step   = output x - input x, the actual length of step taken.
   ycoeff = vector (with length 6*neq) of coefficient values for the
            quintic Hermite interpolation used by function Yvalue.

  Function Rke is organized so that subsequent calls to continue the
  integration involve little, if any, extra effort.  The value of flag
  must, however, be monitored in order to determine what to do next.
  Specifically, if
      flag = 0, the code may be called again to continue the
                integration another step towards x+h.  You may
                shorten h if you wish; if its sign is changed,
                you must restart the integration (first = true).
      flag = 1, the code has attempted 500 steps: either
                tol/threshold are not appropriate or possibly the
                solution does not exist beyond x.  You may alter
                tol or threshold.  If you want to continue, set
                first = true and call rke again; an additional 500
                steps will be allowed.
      flag = 2, the error parameters tol/threshold are too small.
                If a solution component is zero, use a positive value
                for the corresponding component of threshold.  To
                continue with increased error parameters, set
                first = true and call Rke again.
      flag < 0, you cannot continue the solution of this problem;
                any attempt to do so will result in a program stop.
/*
    Local variables:
*/
    int failed, i, ix, valid;
    double alpha, beta, gamma, err, h12, h180, tolaim, wt, xmid, xx;
    static double *k0, *k1, *k2, *k3, *k4, *k5, *k6, *k7, *k8, *ymid,
                  *y4th, *y5th;
    static double zero = 0.0, hundredth = 0.01, tenth = 0.1, fifth = 0.2,
                  half = 0.5, six_tenths = 0.6, c1 = 1.0, c2 = 2.0,
                  c4 = 4.0, c6 = 6.0, c8 = 8.0, c10 = 10.0, c12 = 12.0,
                  c14 = 14.0, c15 = 15.0, c24 = 24.0, c32 = 32.0,
                  c64 = 64.0, c92 = 92.0, c96 = 96.0, c121 = 121.0,
                  c144 = 144.0, c180 = 180.0;
    static int max_steps = 500;
    static int true = 1, false = 0;
    static int alloc = 0, num_steps;
/*********************************************************************
*   Machine dependent constant:                                      *
*   Set u to the appropriate unit roundoff level for your machine;   *
*   here we assume the correct value is built-in.                    */
          static double u = DBL_EPSILON;
/*********************************************************************/

     if (*first)
    {
/*
       First pass; test the input data.
*/
       valid = true;
       if (neq < 1) valid = false;
       if ((tol > hundredth) ||  (tol < c10*u)) valid = false;
       for (i = 0; i < neq; i++) valid = valid * (threshold[i] >= zero);
       for (i = 0; i < neq; i++)
         if ((abs_d(y[i]) == zero) && (threshold[i] == zero)) valid = false;
       if (!valid)
       {
          *flag = -1;
          return;
       }
       if (!alloc)
       {
/*
          Allocate space for k0, k1, ..., k8, etc.
*/
          k0 = (double*) malloc(neq*sizeof(double));
          k1 = (double*) malloc(neq*sizeof(double));
          k2 = (double*) malloc(neq*sizeof(double));
          k3 = (double*) malloc(neq*sizeof(double));
          k4 = (double*) malloc(neq*sizeof(double));
          k5 = (double*) malloc(neq*sizeof(double));
          k6 = (double*) malloc(neq*sizeof(double));
          k7 = (double*) malloc(neq*sizeof(double));
          k8 = (double*) malloc(neq*sizeof(double));
          y4th = (double*) malloc(neq*sizeof(double));
          y5th = (double*) malloc(neq*sizeof(double));
          ymid = (double*) malloc(neq*sizeof(double));
          alloc = true;
          if (ymid == NULL)
          {
             *flag = -2;
             return;
          }
       }
/*
       Initialize.
*/
       (*f)(*x, y, k0);
       num_steps = 0;
    }
    else
    {
/*
       Continuation; test some input data.
*/
       if (*flag < 0) return;
       valid = true;
       if ((tol > hundredth) || (tol < c10*u)) valid = false;
/*
       Although it is allowed that threshold can be altered between calls,
       the signs of the components are checked only on the first call.
*/
       if (!valid)
       {
          *flag = -1;
          return;
       }
    }
/*
    Start a new step here.
*/
    tolaim = six_tenths*tol;
    failed = false;
/*
    Begin another attempt after a step failure here.
*/
loop: if (num_steps >= max_steps)
    {
       *flag = 1;
       return;
    }
/*
    Use England to take the basic England 4th order step to x+h/2.
*/
    England(f, *x, y, k0, k1, k2, k3, *h, ymid, neq);
    xmid = *x+half* *h;
    (*f)(xmid, ymid, k4);
/*
    Use England to take basic England 4th order step to x+h.
*/
    England(f, xmid, ymid, k4, k5, k6, k7, *h, y4th, neq);
/*
    Calculate slopes at x+h for the 5th order solution.
*/
    h12 = *h/c12;
    for (i = 0; i < neq; i++)
      y5th[i] = y[i]+h12*(-k0[i]-c96*k1[i]+c92*k2[i]-c121*k3[i]+
                c144*k4[i]+c6*k5[i]-c12*k6[i]);
    xx = *x+ *h;
    (*f)(xx, y5th, k8);
    h180 = *h/c180;
    for (i = 0; i < neq; i++)
      y5th[i] = y[i]+h180*(c14*k0[i]+c64*k2[i]+c32*k3[i]-c8*k4[i]+
                c64*k6[i]+c15*k7[i]-k8[i]);
/*
    Form err, the weighted maximum norm of the estimated local error.
*/
    err = zero;
    for (i = 0; i < neq; i++)
      if (y4th[i] != y5th[i])
      {
         wt = c2*(abs_d(y[i])+abs_d(ymid[i]))+abs_d(y4th[i])+abs_d(y5th[i]);
         wt = max_d(threshold[i], wt/c6);
         err = max_d(err, abs_d(y5th[i]-y4th[i])/wt);
      }
    num_steps++;
    if (err > tol)
    {
/*
       Failed step.
*/
       if (*first)
          alpha = tenth;
       else
       {
          if (!failed)
          {
             failed = true;
             alpha = pow(tolaim/err, fifth);
             alpha = max_d(tenth, alpha);
          }
          else
             alpha = half;
       }
       *h = alpha* *h;
       if (abs_d(*h) >= c24*u*max_d(abs_d(*x), abs_d(*x+*h))) goto loop;
       *flag = 2;
    }
    else
    {
/*
       Successful step.
*/
       *x += *h;
       (*f)(*x, y5th, k8);
       *step = *h;
       for (i = 0; i < neq; i++)
       {
          alpha = (k0[i]-k8[i])* *step;
          beta = y5th[i]-c2*ymid[i]+y[i];
          gamma = y5th[i]-y[i];
          ix = 6*i;
          ycoeff[ix] = ymid[i];
          ycoeff[1+ix] = k4[i]* *step;
          ycoeff[2+ix] = c4*beta+half*alpha;
          ycoeff[3+ix] = c10*gamma- *step*(k0[i]+c8*k4[i]+k8[i]);
          ycoeff[4+ix] = -c8*beta-c2*alpha;
          ycoeff[5+ix] = -c24*gamma+c4* *step*(k0[i]+c4*k4[i]+k8[i]);
          y[i] = y5th[i];
          k0[i] = k8[i];
       }
       *first = false;
       beta = pow(err/tolaim, fifth);
       alpha = c1/max_d(tenth, beta);
       if (failed) alpha = min_d(c1, alpha);
       *h = alpha* *h;
       *flag = 0;
    }
    return;
}
/* ------------------------------------------------------------------------ */
void England(f, x, y, fin, fout1, fout2, fout3, h, ynew, neq)
   void (*f)( );
   int neq;
   double x, y[], fin[], fout1[], fout2[], fout3[], h, ynew[];
{
    double h2,h4,h8,h12;
    static double eighth = 0.125, quarter = 0.25, half = 0.5,
                  two = 2.0, four = 4.0, twelve = 12.0;
    int i;
/*
    Use the basic England formulas to compute ynew, the 4th order
    solution at x+h/2.
*/
    h4 = quarter*h;
    for (i = 0; i < neq; i++) ynew[i] = y[i]+h4*fin[i];
    (*f)(x+h4, ynew, fout1);

    h8 = eighth*h;
    for (i = 0; i < neq; i++) ynew[i] = y[i]+h8*(fin[i]+fout1[i]);
    (*f)(x+h4, ynew, fout2);

    h2 = half*h;
    for (i = 0; i < neq; i++) ynew[i] = y[i]+h2*(-fout1[i]+two*fout2[i]);
    (*f)(x+h2, ynew, fout3);

    h12 = h/twelve;
    for (i = 0; i < neq; i++)
         ynew[i] = y[i]+h12*(fin[i]+four*fout2[i]+fout3[i]);
}
/* ------------------------------------------------------------------------ */
void Yvalue(neq, x, step, ycoeff, point, u)
   int neq;
   double x, step, ycoeff[], point, u[];
{
/*
  Evaluate the quintic Hermite interpolants based on output from function Rke.

  Input parameters:
    neq    = number of components in vector y.
    x      = output from Rke (final value for independent variable).
    step   = output from Rke (total step taken).
    ycoeff = output from Rke (coefficients of quintics).
    point  = point at which evaluation is to be made.

  Output parameters:
    u      = vector of solution components at point.

  Local variables    */
   int i,ix;
   double z;
   static double half = 0.5;
/*
   Transform from [x-step,x] to [-0.5,0.5], then evaluate.
*/
   z = (point-x)/step+half;
   for (i = 0; i < neq; i++)
   {
     ix = 6*i;
     u[i] = ycoeff[ix]+z*(ycoeff[1+ix]+z*(ycoeff[2+ix]+z*
           (ycoeff[3+ix]+z*(ycoeff[4+ix]+z*ycoeff[5+ix]))));
   }
}
