!     Last change:  LFS  26 Dec 97    9:56 am
MODULE SAP_WP
! The KIND of the working precision, WP, is defined here in a general way.
! To specify a particular KIND, set WP appropriately in this one statement.

  IMPLICIT NONE
  ! WP is the working precision.
  INTEGER, PARAMETER :: WP = SELECTED_REAL_KIND(10,50)

END MODULE SAP_WP

MODULE SAP_CODES
!===============================================================
! This module contains all the subprograms provided for the text
! Lawrence Shampine, Richard Allen, Steven Pruess, Fundamentals
! of Numerical Computing.
!
! Last change:  January, 1998
!===============================================================

  CONTAINS
 
!===============================================================
! Programs for Chapter 2:
!===============================================================

  SUBROUTINE FACTOR(A,NEQ,FLAG,PIVOTS,COND)
  ! FACTOR decomposes the matrix A using Gaussian elimination and
  ! optionally estimates its condition number.  FACTOR is used in
  ! conjunction with SOLVE to solve A*X = B, a system of NEQ linear
  ! equations in NEQ unknowns.  It is sometimes convenient for the
  ! matrix to be defined as the portion A(1:NEQ,1:NEQ) of a larger
  ! matrix, and similarly for the vector B. FACTOR/SOLVE are coded
  ! so that this is possible.
  !
  ! Input arguments:
  !    A       = real matrix A(1:NEQ,1:NEQ) to be factored.
  !    NEQ     = the number of equations to be solved, an integer.
  !
  ! Output arguments:
  !    A       = contains the upper triangular matrix U in its upper
  !              portion (by rows) and a permuted version of a lower
  !              triangular matrix (I - L).  The factorization is
  !              such that (permutation matrix)*A = L*U.
  !   FLAG     = an integer that reports success or the reason for
  !              failure.  FLAG = 0 indicates success. If FLAG > 0,
  !              a zero pivot occurred at equation number FLAG and
  !              the computation was terminated.  FLAG = -1 means
  !              there was an input error: either NEQ <= 0 or A does
  !              not have at least NEQ rows and columns.
  !   PIVOTS   = an integer vector of at least NEQ entries that
  !              records row interchanges. The entry PIVOTS(NEQ) =
  !              (-1)**(number of row interchanges).
  !
  !  When FLAG > 0, the determinant of A is 0 and when FLAG = 0,
  !              det(A) = pivots(NEQ) * A(1,1) *  ... * A(NEQ,NEQ).
  !
  !  Optional output argument:
  !   COND     = when FLAG >= 0, an estimate of the condition number
  !              of A in the infinity norm.  COND is a real number.

    USE SAP_WP
    IMPLICIT NONE

    INTEGER, INTENT(IN) :: NEQ
    INTEGER, INTENT(OUT) :: FLAG
    INTEGER, DIMENSION(NEQ), INTENT(OUT) :: PIVOTS
    REAL (KIND = WP), DIMENSION(:,:), INTENT(IN OUT) :: A
    REAL (KIND = WP), OPTIONAL, INTENT(OUT) :: COND

    INTEGER :: I, K, M
    INTEGER, DIMENSION(1) :: OCCURRED
    REAL (KIND = WP) :: ANORM, BIGGEST, DNORM, EK, YNORM, T
    REAL (KIND = WP), DIMENSION(NEQ) :: D, Y

    IF ((NEQ <= 0) .OR. &
        (SIZE(A,DIM=1) < NEQ) .OR. (SIZE(A,DIM=2) < NEQ)) THEN
      FLAG = -1
      RETURN
    END IF
    FLAG = 0
    PIVOTS(NEQ) = 1

    IF (PRESENT(COND)) THEN
      ! Initialize COND for A that is numerically singular.
      COND = HUGE(1.0_WP)

      ! Compute the infinity norm of A before the matrix is
      ! overwritten by its factorization.
      ANORM = MAXVAL(SUM(ABS(A(1:NEQ,1:NEQ)),DIM=2))
    END IF

    IF (NEQ == 1) THEN
      IF (ABS(A(1,1)) <= 0.0_WP) THEN
        FLAG = 1
      ELSE IF (PRESENT(COND)) THEN
        COND = 1.0_WP
      END IF
      RETURN
    END IF

    ! Gaussian elimination with partial pivoting.
    DO K = 1,NEQ-1

      ! Determine the row M containing the largest element in
      ! magnitude to be used as a pivot and its magnitude BIGGEST.
      OCCURRED = MAXLOC(ABS(A(K:NEQ,K)))
      M = OCCURRED(1) + K - 1
      BIGGEST = ABS(A(M,K))

      ! If all possible pivots are 0, A is numerically singular.
      IF (BIGGEST <= 0.0_WP) THEN
        FLAG = K
        RETURN
      END IF
      PIVOTS(K) = M
      IF (M /= K) THEN
        ! Interchange the current row K with the pivot row M.
        D(K:NEQ) = A(K,K:NEQ)
        A(K,K:NEQ) = A(M,K:NEQ)
        A(M,K:NEQ) = D(K:NEQ)
        PIVOTS(NEQ) = - PIVOTS(NEQ)
      END IF

      ! Eliminate subdiagonal entries of column K.
      DO I = K+1,NEQ
        T = A(I,K)/A(K,K)
        A(I,K) = - T
        IF (ABS(T) > 0.0_WP) THEN
          A(I,K+1:NEQ) = A(I,K+1:NEQ) - T*A(K,K+1:NEQ)
        END IF
      END DO

    END DO

    IF (ABS(A(NEQ,NEQ)) <= 0.0_WP) THEN
      FLAG = NEQ
      RETURN
    END IF

    IF (PRESENT(COND)) THEN
      ! Estimate the condition number of A by computing the infinity
      ! norm of A directly and a lower bound for the norm of A**(-1).
      ! A lower bound for the norm of A**(-1) is provided by the ratio
      ! norm(Y)/norm(D) for any vectors such that A*Y = D and D /= 0.
      ! A "large" ratio is obtained by computing Y as one iteration of
      ! inverse iteration for the smallest singular value of A, i.e.,
      ! by solving for Y such that (transpose(A)*A)*Y = E.  This exploits
      ! the fact that an LU decomposition of A can be used to solve the
      ! linear system transpose(A)*D = E as well as A*Y = D.  The entries
      ! of E are +1 or -1 with the sign chosen during the computation of
      ! D to increase the size of the entry of D and so make a "large"
      ! lower bound for the norm of A**(-1) more likely.

      ! Solve transpose(A)*D = E using the decomposition of A.
      D(1) = -1.0_WP/A(1,1)
      DO K = 2,NEQ
        T = DOT_PRODUCT(A(1:K-1,K),D(1:K-1))
        IF (T < 0.0_WP) THEN
          EK = -1.0_WP
        ELSE
          EK =  1.0_WP
        END IF
        D(K) = - (EK + T)/A(K,K)
      END DO
      DO K = NEQ-1,1,-1
        D(K) = D(K) + DOT_PRODUCT(A(K+1:NEQ,K),D(K+1:NEQ))
        M = PIVOTS(K)
        T = D(M)
        D(M) = D(K)
        D(K) = T
      END DO

      ! Solve A*Y = D.
      Y = SOLVE(A,NEQ,PIVOTS,D)

      ! Compute the infinity norms of the vectors.
      DNORM = MAXVAL(ABS(D))
      YNORM = MAXVAL(ABS(Y))

      COND = MAX(ANORM*YNORM/DNORM,1.0_WP)

    END IF
    RETURN

  END SUBROUTINE FACTOR


  FUNCTION SOLVE(A,NEQ,PIVOTS,B) RESULT(X)
  ! SOLVE solves A*X = B, a system of NEQ linear equations in NEQ
  ! unknowns using the decomposition obtained from a successful call
  ! to FACTOR.
  !
  ! Input arguments:
  !    A           = output of FACTOR.  Triangular decomposition
  !                  of the coefficient matrix.
  !    NEQ         = the number of equations to be solved.
  !    PIVOTS      = output of FACTOR. Record of row interchanges.
  !    B           = right hand side vector B, a real vector of at
  !                  least NEQ entries.
  !
  ! Output argument:
  !    X           = solution vector of the same size and type as B.

    USE SAP_WP
    IMPLICIT NONE

    INTEGER, INTENT(IN) :: NEQ
    INTEGER, DIMENSION(:), INTENT(IN) :: PIVOTS
    REAL (KIND = WP), DIMENSION(:,:), INTENT(IN) :: A
    REAL (KIND = WP), DIMENSION(:), INTENT(IN) :: B
    REAL (KIND = WP), DIMENSION(SIZE(B)) :: X

    INTEGER :: I, K, M
    REAL (KIND = WP) :: T

    X = B

    IF (NEQ == 1) THEN

      X(1) = X(1)/A(1,1)

    ELSE

      ! Forward elimination.
      DO K = 1,NEQ-1
        M = PIVOTS(K)
        T = X(M)
        X(M) = X(K)
        X(K) = T
        X(K+1:NEQ) = X(K+1:NEQ) + A(K+1:NEQ,K)*X(K)
      END DO

      ! Back substitution.
      X(NEQ) = X(NEQ)/A(NEQ,NEQ)
      DO I = NEQ-1,1,-1
        X(I) = (X(I) - DOT_PRODUCT(A(I,I+1:NEQ),X(I+1:NEQ)))/A(I,I)
      END DO

    END IF
    RETURN

  END FUNCTION SOLVE

!===============================================================
! Programs for Chapter 3:
!===============================================================

  SUBROUTINE SPCOEF(N,X,F,B,C,D,FLAG)
  ! Calculate coefficients defining a smooth cubic interpolatory spline S.
  !
  ! Input arguments:
  !    N   = number of data points.
  !    X   = vector of N values of the independent variable ordered
  !          so that  X(1) < X(2) < ... < X(N).
  !    F   = vector of values of the dependent variable.
  !
  ! Output arguments:
  !    B   = vector of S'(X(I)) values.
  !    C   = vector of S''(X(I))/2 values.
  !    D   = vector of S'''(X(I)+)/6 values (I < N).
  !   FLAG =  0  normal return.
  !        = -1  one of X, F, B, C, or D has fewer than N entries.
  !        = -2  X vector is incorrectly ordered.
  !
  ! X, F, B, C, and D are real vectors dimensioned at least N in
  ! the calling program.  N and FLAG are integers.

    USE SAP_WP
    IMPLICIT NONE
    INTEGER, INTENT(IN) :: N
    INTEGER, INTENT(OUT) :: FLAG
    REAL (KIND = WP), DIMENSION(:), INTENT(IN) :: X, F
    REAL (KIND = WP), DIMENSION(:), INTENT(OUT) :: B, C, D

    INTEGER :: I, K
    REAL (KIND = WP) :: FP1, FPN, P
    REAL (KIND = WP), DIMENSION(N-1) :: DIFF_F, H

    FLAG = 0

    IF ((SIZE(X,DIM=1) < N) .OR. (SIZE(F,DIM=1) < N) .OR. &
        (SIZE(B,DIM=1) < N) .OR. (SIZE(C,DIM=1) < N) .OR. &
        (SIZE(D,DIM=1) < N)) THEN
      FLAG = -1
      RETURN
    END IF

    H = DIFF(X)
    IF (ANY(H <= 0)) THEN
      FLAG = -2
      RETURN
    END IF

    ! Calculate coefficients for the tridiagonal system: store
    ! sub-diagonal in B, diagonal in D, difference quotient in C.

    B(1:N-1) = H
    DIFF_F = DIFF(F)
    C(1:N-1) = DIFF_F/H
    IF (N == 2) THEN
      B(1) = C(1)
      C(1) = 0.0_WP
      D(1) = 0.0_WP
      B(2) = B(1)
      C(2) = 0.0_WP
      RETURN
    END IF
    D(1) = 2.0_WP*B(1)
    DO I = 2,N-1
      D(I) = 2.0_WP*(B(I) + B(I-1))
    END DO
    D(N) = 2.0_WP*B(N-1)

    ! Calculate estimates for the end slopes using polynomials
    ! that interpolate the data nearest the end.
    FP1 = C(1) - B(1)*(C(2) - C(1))/(B(1) + B(2))
    IF (N > 3) THEN
      FP1 = FP1 + B(1)*((B(1) + B(2))*(C(3) - C(2))/ &
           (B(2) + B(3)) - C(2) + C(1))/(X(4) - X(1))
    END IF
          
    FPN = C(N-1) + B(N-1)*(C(N-1) - C(N-2))/(B(N-2) + B(N-1))
    IF (N > 3) THEN
      FPN = FPN + B(N-1)*(C(N-1) - C(N-2) - (B(N-2) + B(N-1))* &
           (C(N-2) - C(N-3))/(B(N-2) + B(N-3)))/(X(N) - X(N-3))
    END IF

    ! Calculate the right hand side and store it in C.
    C(N) = 3.0_WP*(FPN - C(N-1))
    DO I = N-1,2,-1
      C(I) = 3.0_WP*(C(I) - C(I-1))
    END DO
    C(1) = 3.0_WP*(C(1) - FP1)

    ! Solve the tridiagonal system.
    DO K = 2,N
      P = B(K-1)/D(K-1)
      D(K) = D(K) - P*B(K-1)
      C(K) = C(K) - P*C(K-1)
    END DO
    C(N) = C(N)/D(N)
    DO K = N-1,1,-1
      C(K) = (C(K) - B(K)*C(K+1))/D(K)
    END DO

    ! Calculate the coefficients defining the spline.
    D(1:N-1) = DIFF(C)/(3.0_WP * H)
    B(1:N-1) = DIFF_F/H - H*(C(1:N-1) + H*D(1:N-1))
    B(N) = B(N-1) + H(N-1)*(2.0_WP*C(N-1) + H(N-1)*3.0_WP*D(N-1))
    RETURN

    CONTAINS

    FUNCTION DIFF(V)
    ! Auxiliary function to compute the forward difference
    ! of data stored in a vector V.

      USE SAP_WP
      IMPLICIT NONE
      REAL (KIND = WP), DIMENSION(:), INTENT(IN) :: V
      REAL (KIND = WP), DIMENSION(SIZE(V)-1) :: DIFF

      INTEGER :: N

      N = SIZE(V)
      DIFF = V(2:N) - V(1:N-1)
      RETURN
    END FUNCTION DIFF

  END SUBROUTINE SPCOEF


  SUBROUTINE SVALUE(X,F,B,C,D,T,FLAG,S)
  ! Evaluate the spline S at T using coefficients from SPCOEF.
  ! There are N nodes ordered so that X(1) < X(2) < ... < X(N).
  !
  ! Input arguments:
  !    X, F, B, C, D are defined as in SPCOEF.
  !    T             point where the spline S is to be evaluated.
  !
  ! Output arguments:
  !    S        = value of spline at T.
  !    FLAG     =  0  normal return
  !             =  1  T < X(1)
  !             =  2  T > X(N)
  !
  ! T and S are real scalars, FLAG an integer.

    USE SAP_WP
    IMPLICIT NONE

    INTEGER, INTENT(OUT) :: FLAG
    REAL (KIND = WP), INTENT(IN) :: T
    REAL (KIND = WP), DIMENSION(:), INTENT(IN) :: X, F, B, C, D
    REAL (KIND = WP), INTENT(OUT) :: S

    INTEGER, SAVE :: LAST_INTERVAL = 1
    INTEGER :: INTERVAL, J, N
    REAL (KIND = WP) :: DT

    N = SIZE(X,DIM=1)

    ! Search for correct interval for T.
    IF      (T < X(1)) THEN
      FLAG = 1
      INTERVAL = 1
    ELSE IF (T < X(N)) THEN       ! X(1) <= T < X(N).
      FLAG = 0
      IF (T >= X(LAST_INTERVAL)) THEN
        DO J = LAST_INTERVAL,N-1
          IF (T < X(J+1)) THEN
            INTERVAL = J
            EXIT
          END IF
        END DO
      ELSE
        DO J = LAST_INTERVAL-1,1,-1
          IF (T >= X(J)) THEN
            INTERVAL = J
            EXIT
          END IF
        END DO
      END IF
    ELSE IF (T > X(N)) THEN
      FLAG = 2
      INTERVAL = N - 1
    ELSE                          ! T = X(N)
      FLAG = 0
      INTERVAL = N - 1
    END IF

    LAST_INTERVAL = INTERVAL

    ! Evaluate cubic polynomial.
    DT = T - X(INTERVAL)
    S = F(INTERVAL) + DT*(B(INTERVAL) + DT*(C(INTERVAL) + DT*D(INTERVAL)))
    RETURN

  END SUBROUTINE SVALUE

!===============================================================
! Programs for Chapter 4:
!===============================================================

  SUBROUTINE ZERO(F,B,C,ABSERR,RELERR,RESIDUAL,FLAG)
  ! ZERO computes a root of the nonlinear equation F(X) = 0 when F(X)
  ! is a continuous real function of a single real variable X.  The
  ! method used is a combination of bisection and the secant rule.
  !
  !  Input arguments:
  !    F      = name of the function subprogram defining F(X).
  !             This subprogram should have the form
  !                 FUNCTION F(X)
  !                   USE SAP_WP
  !                   IMPLICIT NONE
  !                   REAL (KIND = WP), INTENT(IN) :: X
  !                   REAL (KIND = WP) :: F
  !                   F = ...
  !                   RETURN
  !                 END FUNCTION F
  !             An explicit interface should be provided
  !             in the calling program.
  !    B,C    = values of X such that F(B)*F(C) <= 0.
  ! ABSERR,RELERR = absolute and relative error tolerances.
  !            The stopping criterion is
  !                 ABS(B - C) <= 2*MAX(ABSERR,ABS(B)*RELERR).
  !
  !  Output arguments:
  !    B,C      = see FLAG returns.
  !    RESIDUAL = value of final residual F(B).
  !    FLAG     = 0 for normal return: F(B) = 0 or F(B)*F(C) < 0, the
  !                 stopping criterion is met, and ABS(F(B)) <= ABS(F(C)).
  !             = 1 if too many (500) function evaluations were made.
  !             = 2 if ABS(F(B)) is more than 100 times the larger
  !                 of the residuals for the input B and C.  B probably
  !                 approximates a pole rather than a root.
  !             =-1 if ABSERR < 0 or RELERR is too small for the
  !                 working precision.
  !             =-2 if F(B)*F(C) > 0 on input.
  !
  ! B, C, ABSERR, RELERR and the results F and RESIDUAL of evaluating
  ! the function are real scalars. FLAG is an integer.

    USE SAP_WP
    IMPLICIT NONE
    INTERFACE
      FUNCTION F(X)
        USE SAP_WP
        IMPLICIT NONE
        REAL (KIND = WP), INTENT(IN) :: X
        REAL (KIND = WP) :: F
      END FUNCTION F
    END INTERFACE
    INTEGER, INTENT(OUT) :: FLAG
    REAL (KIND = WP), INTENT(IN) :: ABSERR, RELERR
    REAL (KIND = WP), INTENT(IN OUT) :: B, C
    REAL (KIND = WP), INTENT(OUT) :: RESIDUAL

    INTEGER, PARAMETER :: MAXF = 500
    REAL (KIND = WP), PARAMETER :: UROUND = EPSILON(1.0_WP)

    INTEGER :: COUNT, NUMF
    LOGICAL :: BISECT
    REAL (KIND = WP), SAVE :: RES
    REAL (KIND = WP) :: A, ACMB, CMB, FA, FB, FC, P, Q, TOL, WIDTH
  
    ! Test the input tolerances.
    IF ((RELERR < 10.0_WP*UROUND) .OR. (ABSERR < 0.0_WP)) THEN
      FLAG = -1
      RETURN
    END IF

    ! Initialization.
    COUNT = 0
    WIDTH = ABS(B - C)
    A = C
    FA = F(A)
    NUMF = 1
    IF (ABS(FA) <= 0.0_WP) THEN
      FLAG = 0
      B = A
      RESIDUAL = 0.0_WP
      RETURN
    END IF
    FB = F(B)
    NUMF = 2
    IF (ABS(FB) <= 0.0_WP) THEN
      FLAG = 0
      RESIDUAL = 0.0_WP
      RETURN
    END IF
    IF (SIGN(1.0_WP,FA)*SIGN(1.0_WP,FB) > 0.0_WP) THEN
      FLAG = -2
      RETURN
    END IF
    RES = MAX(ABS(FA),ABS(FB))
    FC = FA
  
    DO  ! Main loop

      ! Interchange B and C so that ABS(F(B)) <= ABS(F(C)).
      IF (ABS(FC) < ABS(FB)) THEN
        A = B
        FA = FB
        B = C
        FB = FC
        C = A
        FC = FA
      END IF
      CMB = 0.5_WP*(C - B)
      ACMB = ABS(CMB)
      TOL = MAX(ABSERR,ABS(B)*RELERR)

      ! Test the stopping criterion and function count.
      IF (ACMB <= TOL) THEN
        RESIDUAL = FB
        IF(ABS(RESIDUAL) > 100.0_WP*RES) THEN
          FLAG = 2
        ELSE
          FLAG = 0
        END IF
        RETURN
      END IF
      IF (NUMF >= MAXF) THEN
        FLAG = 1
        RETURN
      END IF

      ! Calculate new iterate implicitly as B + P/Q after arranging
      ! that P >= 0.  The implicit form is used to prevent overflow.
      P = (B - A)*FB
      Q = FA - FB
      IF (P < 0.0_WP) THEN
        P = -P
        Q = -Q
      END IF

      ! Update A.  Check if the length of the interval containing
      ! a zero is being reduced at a reasonable rate, and if not,
      ! bisect until it is.
      A = B
      FA = FB
      COUNT = COUNT + 1
      BISECT = .FALSE.
      IF (COUNT >= 4) THEN
        IF (8.0_WP*ACMB >= WIDTH) THEN
          BISECT = .TRUE.
        ELSE
          COUNT = 0
          WIDTH = ACMB
        END IF
      END IF

      ! Test for too small a change.
      IF (P <= ABS(Q)*TOL) THEN
        B = B + SIGN(TOL,CMB) ! Smallest increment allowed.
      ELSE

        ! Root ought to be between B and (C+B)/2.
        IF (P < CMB*Q) THEN
          B = B + P/Q     ! Use secant rule.
        ELSE
          BISECT = .TRUE. ! Use bisection.
        END IF
      
      END IF
      
      IF (BISECT) THEN
        B = C - CMB
      END IF
      
      ! The new iterate B has been computed.
      FB = F(B)
      NUMF = NUMF + 1
      IF (ABS(FB) <= 0.0_WP) THEN
        FLAG = 0
        C = B
        RESIDUAL = FB
        RETURN
      END IF
      IF (SIGN(1.0_WP,FB)*SIGN(1.0_WP,FC) > 0.0_WP) THEN
        C = A
        FC = FA
      END IF
    
    END DO  ! Main loop
    RETURN
  
  END SUBROUTINE ZERO

!===============================================================
! Programs for Chapter 5:
!===============================================================

  SUBROUTINE ADAPT(F,A,B,ABSERR,RELERR,ANSWER,ERREST,FLAG)
  ! ADAPT estimates the definite integral of F(X) from A to B
  ! using an adaptive quadrature scheme based on Gauss-Kronrod
  ! (3,7) formulas.
  !
  !  Input arguments:
  !     F      = name of the function subprogram defining F(X).
  !              This subprogram should have the form
  !                 FUNCTION F(X)
  !                   USE SAP_WP
  !                   IMPLICIT NONE
  !                   REAL (KIND = WP), INTENT(IN) :: X
  !                   REAL (KIND = WP) :: F
  !                   F = ...
  !                   RETURN
  !                 END FUNCTION F
  !              An explicit interface should be provided
  !              in the calling program.  
  !     A, B   = end points of integration interval.
  !     ABSERR = absolute error tolerance desired.
  !     RELERR = relative error tolerance desired.
  !
  !  Output arguments:
  !     ANSWER = computed estimate of the integral.
  !     ERREST = estimate of the absolute error in ANSWER.
  !     FLAG   = 0  for normal return.
  !            = 1  insufficient storage in queue (120).
  !            = 2  too many function evaluations (3577).
  !            =-1  if ABSERR <= 0  or RELERR is too small for the
  !                 working precision.
  !
  ! A, B, ABSERR, RELERR and the results ANSWER and ERREST are real
  ! scalars. FLAG is an integer.

    USE SAP_WP
    IMPLICIT NONE
    INTERFACE
      FUNCTION F(X)
        USE SAP_WP
        IMPLICIT NONE
        REAL (KIND = WP), INTENT(IN) :: X
        REAL (KIND = WP) :: F
      END FUNCTION F
    END INTERFACE
    INTEGER, INTENT(OUT) :: FLAG
    REAL (KIND = WP), INTENT(IN) :: A, B, ABSERR, RELERR
    REAL (KIND = WP), INTENT(OUT) :: ANSWER, ERREST

    INTEGER, PARAMETER :: MAXF = 3577, MAXQ = 120
    REAL (KIND = WP), PARAMETER :: UROUND = EPSILON(1.0_WP)

    INTEGER :: BOTTOM, LENGTH, NUMF, TOP
    REAL (KIND = WP) :: ALPHA, BETA, E, EL, ER, H, Q, QL, QR, TOL
    REAL (KIND = WP), DIMENSION(MAXQ,4) :: QUEUE

    ! Test input data.
    IF ((ABSERR <= 0.0_WP) .OR. (RELERR < 10.0_WP*UROUND)) THEN
      FLAG = -1
      RETURN
    END IF

    ! Initialization.
    LENGTH = 0
    TOP = 1
    BOTTOM = 1
    FLAG = 0
    NUMF = 0

    ! Form an initial approximation ANSWER to the integral over [A,B].
    ! If it is not sufficiently accurate, initialize the queue and
    ! begin the main loop.
    CALL QUAD(A,B,ANSWER,ERREST,NUMF)
    IF(ABS(ERREST) > MAX(ABSERR,RELERR*ABS(ANSWER))) THEN
      CALL ADD(ANSWER,ERREST,A,B,LENGTH,BOTTOM)
    END IF

    ! If the queue is empty, return, else subdivide top entry.
    DO  ! Main loop
      IF (LENGTH == 0) THEN
        RETURN
      END IF
      CALL REMOVE(Q,E,ALPHA,BETA,LENGTH,TOP)
      H = 0.5_WP*(BETA - ALPHA)
      CALL QUAD(ALPHA,ALPHA+H,QL,EL,NUMF)
      CALL QUAD(ALPHA+H,BETA,QR,ER,NUMF)

      ! Update ANSWER and the error estimate.
      ANSWER = ANSWER + ((QL + QR) - Q)
      ERREST = ERREST + ((EL + ER) - E)

      ! Test for failures.
      IF (LENGTH >= MAXQ-1) THEN
        FLAG = 1
        RETURN
      END IF
      IF (NUMF >= MAXF) THEN
        FLAG = 2
        RETURN
      END IF

      ! Test for convergence.
      TOL = MAX(ABSERR,RELERR*ABS(ANSWER))
      IF (ABS(ERREST) <= TOL) THEN
        RETURN
      END IF

      ! Add new subintervals to queue if errors are too big.
      TOL = TOL*H/(B - A)
      IF (ABS(EL) > TOL) THEN
        CALL ADD(QL,EL,ALPHA,ALPHA+H,LENGTH,BOTTOM)
      END IF
      IF (ABS(ER) > TOL) THEN
        CALL ADD(QR,ER,ALPHA+H,BETA,LENGTH,BOTTOM)
      END IF

    END DO  ! Main loop
    RETURN

  
    CONTAINS
      
    SUBROUTINE ADD(Q,E,ALPHA,BETA,LENGTH,BOTTOM)
    ! Add an entry to the end of the queue.

      INTEGER, INTENT(IN OUT) :: LENGTH, BOTTOM
      REAL (KIND = WP), INTENT(IN) :: Q, E, ALPHA, BETA
      
      QUEUE(BOTTOM,1) = Q
      QUEUE(BOTTOM,2) = E
      QUEUE(BOTTOM,3) = ALPHA
      QUEUE(BOTTOM,4) = BETA
      LENGTH = LENGTH + 1
      IF (BOTTOM < MAXQ+1) THEN
        BOTTOM = BOTTOM + 1
      END IF
      IF (BOTTOM == MAXQ+1) THEN
        BOTTOM = 1
      END IF
      RETURN
    
    END SUBROUTINE ADD

    SUBROUTINE REMOVE(Q,E,ALPHA,BETA,LENGTH,TOP)
    ! Delete the top entry in the queue.
    
      INTEGER, INTENT(IN OUT) :: LENGTH, TOP
      REAL (KIND = WP), INTENT(OUT) :: Q, E, ALPHA, BETA

      Q = QUEUE(TOP,1)
      E = QUEUE(TOP,2)
      ALPHA = QUEUE(TOP,3)
      BETA = QUEUE(TOP,4)
      LENGTH = LENGTH - 1
      IF (TOP < MAXQ+1) THEN
        TOP = TOP + 1
      END IF
      IF (TOP == MAXQ+1) THEN
        TOP = 1
      END IF
      RETURN
    
    END SUBROUTINE REMOVE

    SUBROUTINE QUAD(ALPHA,BETA,Q,E,NUMF)
    ! Gauss-Kronrod(3,7) quadrature over (ALPHA,BETA).

      INTEGER, INTENT(IN OUT) :: NUMF
      REAL (KIND = WP), INTENT(IN) :: ALPHA, BETA
      REAL (KIND = WP), INTENT(OUT) :: Q, E
      REAL (KIND = WP) :: F1, F2, F3, H, MIDPT, QKRON
      REAL (KIND = WP), DIMENSION(3) :: X = &
           (/ 0.7745966692414834_WP, 0.9604912687080202_WP, &
              0.4342437493468026_WP /)
      REAL (KIND = WP), DIMENSION(4) :: A = &
           (/ 0.2684880898683334_WP, 0.1046562260264672_WP, &
              0.4013974147759622_WP, 0.4509165386584744_WP /)

      H = 0.5_WP*(BETA - ALPHA)
      MIDPT = ALPHA + H
      F1 = F(MIDPT-H*X(1))
      F2 = F(MIDPT)
      F3 = F(MIDPT+H*X(1))
      Q = H*(5.0_WP*(F1+F3)+8.0_WP*F2)/9.0_WP
      QKRON = H*(A(1)*(F1+F3) + A(4)*F2 + A(2)*(F(MIDPT-H*X(2)) &
              + F(MIDPT+H*X(2))) + A(3)*(F(MIDPT-H*X(3)) + F(MIDPT+H*X(3))))
      E = QKRON - Q
      NUMF = NUMF + 7
      RETURN
    
    END SUBROUTINE QUAD
    
  END SUBROUTINE ADAPT

!===============================================================
! Programs for Chapter 6:
!===============================================================

  SUBROUTINE RKE(F,NEQ,X,Y,YP,H,TOL,THRESHOLD,NASTEPS,FLAG,STEP,YCOEFF)
  ! RKE integrates a system of NEQ first order ordinary differential
  ! equations, y' = f(x,y), over one step using a Runge-Kutta method
  ! due to R. England. YVALUE can be used after any successful step by
  ! RKE to approximate the solution components in the interval [X-STEP,X].
  !
  ! Input arguments:
  !   F     = name of the function to evaluate y' = f(x,y). The function
  !           should have the form
  !              FUNCTION F(X,Y) RESULT(YPRIME)
  !                USE SAP_WP
  !                IMPLICIT NONE
  !                REAL (KIND = WP), INTENT(IN) :: X
  !                REAL (KIND = WP), DIMENSION(:), INTENT(IN) :: Y
  !                REAL (KIND = WP), DIMENSION(SIZE(Y)) :: YPRIME
  !                YPRIME(1) = ...
  !                  ...
  !                YPRIME(NEQ) = ...
  !                RETURN
  !              END FUNCTION F
  !              An explicit interface should be provided
  !              in the calling program.  
  !  NEQ    = number of first order equations in the system.
  !   X     = initial value of the independent variable.
  !   Y     = vector of NEQ solution values at X.
  !   YP    = vector YP = f(X,Y) of NEQ components.  On the first call its
  !           value does not matter, but on subsequent calls, it is to be
  !           the value returned by RKE.
  !   H     = step size for current step. The sign of H determines the
  !           direction of integration.  On the first call to RKE you
  !           must specify H.  After the first call, the code suggests
  !           a suitable H for the next step.
  !   TOL, THRESHOLD = desired tolerances: TOL is a scalar and THRESHOLD
  !           is a vector with NEQ components.  The value of TOL must
  !           be in the interval [ 10*UROUND , 0.01 ] where UROUND is the
  !           unit roundoff for the working precision.  Each component of
  !           THRESHOLD must be non-negative.  On the first call, if some
  !           Y(I) = 0, then THRESHOLD(I) must be positive for that I.  The
  !           convergence criterion is for each I
  !            abs(local error in Y(I)) <= TOL*max(THRESHOLD(I),abs(Y(I)))
  !  NASTEPS = number of steps attempted.  On the first call to RKE you
  !           must set NASTEPS = 0.  On subsequent calls, it is to be the
  !           value returned by RKE.
  !
  !  Output arguments:
  !   X      = the integration has advanced to this value of the independent
  !            variable.
  !   Y      = vector of computed solution values at X.
  !   YP     = vector YP = f(X,Y).
  !   H      = step size suitable for the next step.
  !  NASTEPS = number of steps attempted.
  !   FLAG   = 0  for a successful step.
  !            1  if TOL, THRESHOLD are too small
  !           -1  if TOL or some THRESHOLD(I) has a value not allowed
  !
  !  Optional output arguments:
  !   STEP   = output X - input X, the actual step taken.
  !   YCOEFF = array of coefficient values used by YVALUE. Must be
  !            dimensioned YCOEFF(NEQ,6).
  !
  !  RKE is organized so that subsequent calls to continue the integration
  !  involve little, if any, extra effort.  The value of FLAG must, however,
  !  be monitored in order to determine what to do next.  Specifically, if
  !      FLAG = 0  the code may be called again to continue the
  !                integration another step towards X+H.  You may
  !                shorten H if you wish.  If its sign is changed, you
  !                must restart the integration by setting NASTEPS = 0.
  !      FLAG = 1  the error tolerances TOL/THRESHOLD are too small.
  !                If a solution component is zero, use a positive value
  !                for the corresponding component of THRESHOLD.  To
  !                continue with increased error tolerances, set
  !                NASTEPS = 0 and call RKE again.
  !      FLAG =-1  you must correct invalid values of TOL or THRESHOLD.

    USE SAP_WP
    IMPLICIT NONE
    INTERFACE
      FUNCTION F(X,Y) RESULT(YPRIME)
        USE SAP_WP
        IMPLICIT NONE
        REAL (KIND = WP), INTENT(IN) :: X
        REAL (KIND = WP), DIMENSION(:), INTENT(IN) :: Y
        REAL (KIND = WP), DIMENSION(SIZE(Y)) :: YPRIME
      END FUNCTION F
    END INTERFACE
    INTEGER, INTENT(IN) :: NEQ
    INTEGER, INTENT(IN OUT) :: NASTEPS
    INTEGER, INTENT(OUT) :: FLAG
    REAL (KIND = WP), INTENT(IN) :: TOL
    REAL (KIND = WP), DIMENSION(:), INTENT(IN) :: THRESHOLD
    REAL (KIND = WP), INTENT(IN OUT) :: X, H
    REAL (KIND = WP), DIMENSION(:), INTENT(IN OUT) :: Y, YP
    REAL (KIND = WP), INTENT(OUT), OPTIONAL :: STEP
    REAL (KIND = WP), DIMENSION(:,:), INTENT(OUT), OPTIONAL :: YCOEFF
  
    REAL (KIND = WP), PARAMETER :: UROUND = EPSILON(1.0_WP)

    LOGICAL :: FIRST, FAILED, VALID
    REAL (KIND = WP) :: ERR, TEMP, TOLAIM, XMID
    REAL (KIND = WP), DIMENSION(NEQ) :: ALPHA, BETA, GAMMA, DELTA, &
          K1, K2, K3, K4, K5, K6, K7, K8, KTEMP, YMID, YTEMP, Y4TH,&
          Y5TH, WT
  

    ! Test some input data.
    IF ((TOL > 0.01_WP) .OR. (TOL < 10.0_WP*UROUND)) THEN
      FLAG = -1
      RETURN
    END IF

    TOLAIM = 0.6_WP*TOL
    FIRST = .FALSE.

    IF (NASTEPS == 0) THEN

      !  First call:  test more input data.
      FIRST = .TRUE.
      VALID = NEQ > 1
      IF (ANY(THRESHOLD < 0.0_WP)) THEN
        VALID = .FALSE.
      ELSE IF (ANY((THRESHOLD + ABS(Y)) <= 0.0_WP)) THEN
        VALID = .FALSE.
      END IF
      IF (.NOT. VALID) THEN
        FLAG = -1
        RETURN
      END IF
       
      !  Initialize YP.
      YP = F(X,Y)

    END IF

    FAILED = .FALSE.
    DO  !  Loop until an H is found for which the step succeeds.
    
      ! Compute a 4th order result YMID at XMID = X + H/2.
      YTEMP = Y + 0.25_WP*H*YP
      K1 = F(X+0.25_WP*H, YTEMP)
      YTEMP = Y + 0.125_WP*H*(YP + K1)
      K2 = F(X+0.25_WP*H, YTEMP)
      YTEMP = Y + 0.5_WP*H*(-K1 + 2.0_WP*K2)
      K3 = F(X+0.5_WP*H, YTEMP)
      XMID = X + 0.5_WP*H
      YMID = Y + (H/12.0_WP)*(YP + 4.0_WP*K2 + K3)
      K4 = F(XMID, YMID)

      !  Compute a 4th order result Y4TH at X + H.
      YTEMP = YMID + 0.25_WP*H*K4
      K5 = F(XMID+0.25_WP*H, YTEMP)
      YTEMP = YMID + 0.125_WP*H*(K4 + K5)
      K6 = F(XMID+0.25_WP*H, YTEMP)
      YTEMP = YMID + 0.5_WP*H*(-K5 + 2.0_WP*K6)
      K7 = F(XMID+0.5_WP*H, YTEMP)
      Y4TH = YMID + (H/12.0_WP)*(K4 + 4.0_WP*K6 + K7)
  
      ! Compute a 5th order result Y5TH at X + H.
      YTEMP = Y + (H/12.0_WP)*(-YP - 96.0_WP*K1 + 92.0_WP*K2 &
                - 121.0_WP*K3 + 144.0_WP*K4 + 6.0_WP*K5 - 12.0_WP*K6)
      KTEMP = F(X+H, YTEMP)
      Y5TH = Y + (H/180.0_WP)*(14.0_WP*YP + 64.0_WP*K2 + 32.0_WP*K3 &
               - 8.0_WP*K4 + 64.0_WP*K6 + 15.0_WP*K7 - KTEMP)
  
      !  Form ERR, the weighted maximum norm of the estimated local error.
      WT = (2.0_WP*(ABS(Y) + ABS(YMID)) + ABS(Y4TH) + ABS(Y5TH))/6.0_WP
      WT = MAX(WT, THRESHOLD)
      DELTA = ABS(Y5TH - Y4TH)    ! DELTA(I) is an estimate of the error in Y4TH(I).
      WHERE (WT > 0)              ! WT(I) = 0 => DELTA(I) = 0.
        DELTA = DELTA/WT
      END WHERE
      ERR = MAXVAL(DELTA)
      
      NASTEPS = NASTEPS + 1
      IF (ERR > TOL) THEN
      
        ! Failed step.
        IF (FIRST) THEN
          H = 0.1_WP*H
        ELSE IF (.NOT. FAILED) THEN
          FAILED = .TRUE.
          H = MAX(0.1_WP,(TOLAIM/ERR)**0.2_WP)*H
        ELSE
          H = 0.5_WP*H
        END IF
        IF (ABS(H) < 24.0_WP*UROUND*MAX(ABS(X), ABS(X+H))) THEN
          FLAG = 2
          RETURN
        END IF
    
      ELSE
  
        ! Successful step.
        X = X + H
        K8 = F(X, Y5TH)
        IF (PRESENT(STEP)) THEN
          STEP = H
        END IF
        IF (PRESENT(YCOEFF)) THEN
          ALPHA = H*(YP - K8)
          BETA = Y5TH - 2.0_WP*YMID + Y
          GAMMA = Y5TH - Y
          YCOEFF(:,1) = YMID
          YCOEFF(:,2) = H*K4
          YCOEFF(:,3) = 4.0_WP*BETA + 0.5_WP*ALPHA
          YCOEFF(:,4) = 10.0_WP*GAMMA - H*(YP + 8.0_WP*K4 + K8)
          YCOEFF(:,5) = -8.0_WP*BETA - 2.0_WP*ALPHA
          YCOEFF(:,6) = -24.0_WP*GAMMA + 4.0_WP*H*(YP + 4.0_WP*K4 + K8)
        END IF
        Y = Y5TH
        YP = K8
        TEMP = 1.0_WP/MAX(0.1_WP, (ERR/TOLAIM)**0.2_WP)
        IF (FAILED) THEN
          TEMP = MIN(1.0_WP, TEMP)
        END IF
        H = TEMP*H
        FLAG = 0
        RETURN
      END IF

    END DO
    RETURN
  
  END SUBROUTINE RKE

  FUNCTION YVALUE(XI,X,STEP,YCOEFF) RESULT(Z)
  ! Evaluate the quintic Hermite interpolants based on output from RKE.
  !
  !  Input arguments:
  !    XI      = point at which evaluation is to be made.
  !    X       = output from RKE (final value for independent variable).
  !    STEP    = output from RKE (step taken).
  !    YCOEFF  = output from RKE (coefficients of quintics).
  !
  !  Output argument:
  !    Z       = vector of solution components at XI.

    USE SAP_WP
    IMPLICIT NONE
    REAL (KIND = WP), INTENT(IN) :: XI, X, STEP
    REAL (KIND = WP), DIMENSION(:,:), INTENT(IN) :: YCOEFF
    REAL (KIND = WP), DIMENSION(SIZE(YCOEFF,1)) :: Z
  
    REAL (KIND = WP) :: ARG
  
    !  Transform from [X - STEP, X] to [-0.5, 0.5], then evaluate.
    ARG = 0.5_WP + (XI - X)/STEP
    Z = YCOEFF(:,1) + ARG*(YCOEFF(:,2) + ARG*(YCOEFF(:,3) + &
        ARG*(YCOEFF(:,4) + ARG*(YCOEFF(:,5) + ARG*YCOEFF(:,6)))))
    RETURN
  
  END FUNCTION YVALUE


END MODULE SAP_CODES

