// FUNCTION:     Functions for setting up and evaluating a cubic
//               interpolatory spline.
// AUTHORS:      Lawrence Shampine, Richard Allen, Steven Pruess  for
//               the text  Fundamentals of Numerical Computing
// DATE:         February 27, 1996
// LAST CHANGE:  April 3, 1998

#include "fnc.h"

int Spline_coeff(int n, double x[], double f[], double b[], double c[],
                 double d[])
{
///////////////////////////////////////////////////////////////////////////////
// Calculate coefficients defining a smooth cubic interpolatory spline.

// Input parameters:
//   n   = number of data points.
//   x   = vector of values of the independent variable ordered
//         so that  x[i] < x[i+1]  for all i.
//   f   = vector of values of the dependent variable.
// Output parameters:
//   b   = vector of S'(x[i]) values.
//   c   = vector of S"(x[i])/2 values.
//   d   = vector of S'''(x[i]+)/6 values (i < n).
//   return_value  =  0  normal return;
//                 = -1  input n <= 1;
//                 = -2  x vector is incorrectly ordered.
///////////////////////////////////////////////////////////////////////////////
//  Local variables:
    int i, k;
    double fp1, fpn, h, p;
    const double zero = 0.0, two = 2.0, three = 3.0;

    if (n <= 1) return -1;

//  Calculate coefficients for the tri-diagonal system: store
//  sub-diagonal in b, diagonal in d, difference quotient in c.

    b[0] = x[1] - x[0];
    if (b[0] <= zero) return -2;
    c[0] = (f[1] - f[0]) / b[0];
    if (n == 2)
    {
       b[0] = c[0];
       c[0] = zero;
       d[0] = zero;
       b[1] = b[0];
       c[1] = zero;
       return 0;
    }
    d[0] = two * b[0];
    for (i = 1; i < n-1; i++)
    {
       b[i] = x[i+1] - x[i];
       if (b[i] <= zero) return -2;
       c[i] = (f[i+1] - f[i]) / b[i];
       d[i] = two * (b[i] + b[i-1]);
    }
    d[n-1] = two * b[n-2];

//  Calculate estimates for the end slopes.  Use polynomials
//  interpolating data nearest the end.

    fp1 = c[0] - b[0] * (c[1] - c[0]) / (b[0] + b[1]);
    if (n > 3) fp1 = fp1 + b[0] * ((b[0] + b[1]) * (c[2] - c[1])
       / (b[1] + b[2]) - c[1] + c[0]) / (x[3] - x[0]);
    fpn = c[n-2] + b[n-2] * (c[n-2] - c[n-3]) / (b[n-3] + b[n-2]);
    if (n > 3) fpn = fpn + b[n-2] * (c[n-2] - c[n-3] - (b[n-3] + b[n-2]) 
       * (c[n-3] - c[n-4]) / (b[n-3] + b[n-4])) / (x[n-1] - x[n-4]);

//  Calculate the right-hand-side and store it in c.

    c[n-1] = three * (fpn - c[n-2]);
    for (i = n - 2; i > 0; i--)
        c[i] = three * (c[i] - c[i-1]);
    c[0] = three * (c[0]-fp1);

//  Solve the tridiagonal system.

    for (k = 1; k < n; k++)
    {
        p = b[k-1] / d[k-1];
        d[k] = d[k] - p * b[k-1];
        c[k] = c[k] - p * c[k-1];
    }
    c[n-1] = c[n-1] / d[n-1];
    for (k = n - 2; k >= 0; k--)
        c[k] = (c[k] - b[k] * c[k+1]) / d[k];

//  Calculate the coefficients defining the spline.

    for (i = 0; i < n-1; i++)
    {
        h = x[i+1] - x[i];
        d[i] = (c[i+1] - c[i]) / (three * h);
        b[i] = (f[i+1] - f[i]) / h - h * (c[i] + h * d[i]);
    }
    b[n-1] = b[n-2] + h * (two * c[n-2] + h * three * d[n-2]);
    return 0;
}
///////////////////////////////////////////////////////////////////////////////
int Spline_value(int n, double x[], double f[], double b[], double c[],
                 double d[], double t, int& interval, double& s)
{
//  Evaluate the spline s at t using coefficients from Spline_coeff.

//  Input parameters:
//     n, x, f, b, c, d are defined as in Spline_coeff.
//     t        = point where spline is to be evaluated.
//  Output parameters:
//     interval = index satisfying  x[interval] <= t < x[interval+1]
//                unless t is outside data interval (see flag).
//     s        = value of spline at t.
//     return_value =  0  normal return;
//                  = -1  n <= 1;
//                  =  1  t < x[0];
//                  =  2  t > x[n-1].
///////////////////////////////////////////////////////////////////////////////
//  Local variables:
    static int last_interval = 0;
    int flag, j;
    double dt;

    if (n <= 1) return -1;
    flag = 0;

//  Search for correct interval for t.

    if (t < x[0])
    {
         interval = 0;
         flag = 1;
    }
    if (t > x[n-1])
    {
         interval = n - 2;
         flag = 2;
    }
    if (flag == 0)
    {
       if (t >= x[last_interval])
          for (j = last_interval; j < n - 1; j++)
          {
              if (t < x[j+1])
              {
                  interval = j;
                  break;
              }
          }
       else
          for (j = last_interval - 1; j >= 0; j--)
          {
              if (t >= x[j])
              {
                  interval = j;
                  break;
              }
           }
       last_interval = interval;
    }

//  Evaluate cubic polynomial on [x[interval] , x[interval+1]].

    dt = t - x[interval];
    s = f[interval] + dt * (b[interval] + dt * (c[interval] + dt
         * d[interval]));
    return flag;
}
