// FUNCTION:     Estimate the numerical value of a definite integral.
// AUTHORS:      Lawrence Shampine, Richard Allen, Steven Pruess  for
//               the text  Fundamentals of Numerical Computing
// DATE:         December 8, 1995
// LAST CHANGE:  July 3, 1996
// PURPOSE:      Pointer based implementation of function Adapt.


#include "fnc.h"
#include <float.h>        // for DBL_EPSILON
#include <stdio.h>        // for NULL

int Adapt(double (*f)(double), double a, double b, double abserr,
          double relerr, double& answer, double& errest)
{
///////////////////////////////////////////////////////////////////////////////
//   Adapt estimates the definite integral of f(x) from a to b using an
//   adaptive quadrature scheme based on Gauss-Kronrod (3,7) formulas.
//
//   Input parameters:
//     f      = name of the function defining f(x).
//              This program must have the form:
//              double f(double x)
//              {
//                 ...
//                 return function_value;
//              }
//     a, b   = end points of integration interval.
//     abserr = absolute error tolerance desired.
//     relerr = relative error tolerance desired.
//
//   Output parameters:
//     answer = computed estimate of the integral.
//     errest = estimate of the absolute error in answer.
//     return_value = 0  for normal return;
//                  = 1  insufficient storage for queue
//                  = 2  too many function evaluations (3577);
//                  =-1  invalid input parameters; abserr <= 0  or
//                       relerr too small.
///////////////////////////////////////////////////////////////////////////////
//  Local variables:
      double alpha, beta, e, el, er, h, q, ql, qr, tol;
      int flag, num_func_evals = 0;
      QuadNode* queue;         // Use a pointer based circular queue.
      const double zero = 0.0, half = 0.5, ten = 10.0;
      const int max_func_evals = 3577;
///////////////////////////////////////////////////////////////////////////////
//  Machine dependent constant:                                              //
//                                                                           //
//  Set u to the appropriate unit roundoff level for your machine;           //
//  here we assume the correct value is built-in.                            //
          const double u = DBL_EPSILON;
///////////////////////////////////////////////////////////////////////////////
//  Test input data.

    if ((abserr <= zero) || (relerr < ten * u)) return -1;

//  Initialize circular queue.

    queue = new QuadNode;
    if (queue == NULL) return 1;
    queue->Next = queue;

//  Form an initial approximation answer to the integral over [a,b].
//  If it is not sufficiently accurate, initialize the queue and
//  begin the main loop.

    Quad(f, a, b, q, e, num_func_evals);
    answer = q;
    errest = e;
    if (abs_d(errest) > max_d(abserr, relerr * abs_d(answer)))
    {
       flag = Add(queue, q, e, a, b);
       if (flag != 0) return 1;
    } 

//  Main loop; if queue is empty then return, else subdivide the top entry.

    while (!Empty(queue))
    {
        Remove(queue, q, e, alpha, beta);
        h = half * (beta - alpha);
        Quad(f, alpha, alpha + h, ql, el, num_func_evals);
        Quad(f, alpha + h, beta, qr, er, num_func_evals);

//      Update answer and the error estimate.

        answer += ((ql + qr) - q);
        errest += ((el + er) - e);

//      Test for failures.

        if (num_func_evals >= max_func_evals) return 2;

//      Test for convergence.

        tol = max_d(abserr, relerr * abs_d(answer));
        if (abs_d(errest) <= tol) return 0;

//      Add new subintervals to queue if errors are too big.

        tol = tol * h / (b - a);
        if (abs_d(el) > tol)
        {
           flag = Add(queue, ql, el, alpha, alpha+h);
           if (flag != 0) return 1;
        }
        if (abs_d(er) > tol)
        {
           flag = Add(queue, qr, er, alpha+h, beta);
           if (flag != 0) return 1;
        }
    }
    return 0;
}
///////////////////////////////////////////////////////////////////////////////
int Add(QuadNode* Rear, double q, double e, double alpha, double beta)
{
//  Add an entry to the end of the queue.

    QuadNode* NewNode = new QuadNode;
    if (NewNode == NULL) return -1;

    NewNode->Q = q;
    NewNode->E = e;
    NewNode->alpha = alpha;
    NewNode->beta = beta;

    QuadNode* Temp = Rear->Next;   // Let Temp point to the front for now.
    Rear->Next = NewNode;
    Rear = NewNode;
    Rear->Next = Temp;

    return 0;
}
///////////////////////////////////////////////////////////////////////////////
void Remove(QuadNode* Rear, double& q, double& e, double& alpha, double& beta)
{
//  Retrieve and Delete the first item in the queue.

    QuadNode* NodePtr = Rear->Next;     // Retrieve first item.
    q = NodePtr->Q;
    e = NodePtr->E;
    alpha = NodePtr->alpha;
    beta = NodePtr->beta;
    Rear->Next = NodePtr->Next;         // Delete first item.
    delete NodePtr;
}
///////////////////////////////////////////////////////////////////////////////
int Empty(QuadNode* Rear)
{
//  Check to see if queue is empty.
   
    if (Rear->Next == Rear)
       return 1;
    else
       return 0;
}
///////////////////////////////////////////////////////////////////////////////
void Quad(double (*f)(double), double alpha, double beta, double& q, double& e,
          int& num_func_evals)
{
//  Apply quadrature rule with error estimate over [alpha, beta].

    double f1, f2, f3, h, midpoint, qkronrod;
    const double half = 0.5, five = 5.0, eight = 8.0, nine = 9.0;

//  Gauss-Kronrod(3,7) quadrature over (alpha,beta).

    const double a[4] = {0.2684880898683334, 0.1046562260264672,
                         0.4013974147759622, 0.4509165386584744};
    const double x[3] = {0.7745966692414834, 0.9604912687080202,
                         0.4342437493468026};
    h = half * (beta - alpha);
    midpoint = alpha + h;
    f1 = (*f)(midpoint - h * x[0]);
    f2 = (*f)(midpoint);
    f3 = (*f)(midpoint + h * x[0]);
    q = h * (five * (f1 + f3) + eight * f2) / nine;
    qkronrod = h*(a[0]*(f1+f3) + a[3]*f2 +
               a[1]*((*f)(midpoint-h*x[1]) + (*f)(midpoint+h*x[1])) +
               a[2]*((*f)(midpoint-h*x[2]) + (*f)(midpoint+h*x[2])));
    e = qkronrod - q;
    num_func_evals += 7;
}
