/* FUNCTION:     Find a zero of a single nonlinear equation.
   AUTHORS:      Lawrence Shampine, Richard Allen, Steven Pruess  for
                 the text  Fundamentals of Numerical Computing
   DATE:         August 25, 1995
   LAST CHANGE:  January 29, 1998                                */


#include "fnc.h"

#include <float.h>       /*   for DBL_EPSILON   */

int Zero(f, b, c, abserr, relerr, residual)
   double (*f)( ), *b, *c, abserr, relerr, *residual;
{
/*
Zero computes a root of the nonlinear equation f(x) = 0 when f(x)
is a continuous real function of a single real variable x.  The
method used is a combination of bisection and the secant rule.

  Input parameters:
     f     = function program defining f(x).  This subprogram
             must have the form:
                double f(x);
                   double x;
                {
                   ...
                   return function_value;
                }
     b,c   = values of x such that f(b)*f(c) <= 0.
     abserr,relerr = absolute and relative error tolerances.
             The stopping criterion is:
               abs(b-c) <= 2.0*max(abserr,abs(b)*relerr).
  Output parameters:
     b,c          = see return_value options.
     residual     = value of final residual f(b).
     return_value = 0 for normal return; f(b)*f(c) < 0 and the
                      stopping criterion is met (or f(b) = 0).  b is
                      always selected so that abs(f(b)) <= abs(f(c)).
                  = 1 if too many function evaluations were made; in
                      this version 500 are allowed.
                  = 2 if abs(f(b)) is more than 100 times the larger
                      of the residuals for the input b and c.  b and c
                      probably approximate a pole rather than a root.
                  =-1 if abserr is less than zero, or relerr too small.
                  =-2 if f(b)*f(c) is positive on input.

  Local variables:  */
    double a, acmb, cmb, fa, fb, fc, p, q, tol, width;
    static double initial_residual;
    static double zero = 0.0, half = 0.5, one = 1.0, eight = 8.0,
                  ten = 10.0, hundred = 100.0;
    static int true = 1, false = 0;
    int bisect, count, flag, max_func_evals = 500, num_func_evals;
/***********************************************************************
*   Machine dependent constant:                                        *
*                                                                      *
*   Set u to the appropriate unit roundoff level for your machine;     *
*   here we assume the corect value is built-in.                       */
          static double u = DBL_EPSILON;
/***********************************************************************
    Test the input tolerances.
*/
    if ((relerr < ten*u) || (abserr < zero)) return -1;
/*
    Initialization.
*/
    count = 0;
    width = abs_d(*b-*c);
    a = *c;
    fa = (*f)(a);
    num_func_evals = 1;
    if (abs_d(fa) == zero)
    {
       *b = a;
       *residual = zero;
       return 0;
    }
    fb = (*f)(*b);
    num_func_evals = 2;
    if (abs_d(fb) == zero)
    {
       *residual = zero;
       return 0;
    }
    if (sign(one, fa) == sign(one, fb)) return -2;
    initial_residual = max_d(abs_d(fa),abs_d(fb));
    fc = fa;
    for (;;)
    {
       if (abs_d(fc) < abs_d(fb))
       {
/*
         Interchange b and c so that abs(f(b)) <= abs(f(c)).
*/
          a = *b;
          fa = fb;
          *b = *c;
          fb = fc;
          *c = a;
          fc = fa;
       }
       cmb = half*(*c-*b);
       acmb = abs_d(cmb);
       tol = max_d(abserr, abs_d(*b)*relerr);
/*
       Test the stopping criterion and function count.
*/
       if (acmb <= tol)
       {
          *residual = fb;
          if (abs_d(*residual) > hundred*initial_residual)
             return 2;
          else
             return 0;
       }
       if (num_func_evals >= max_func_evals) return 1;
/*
       Calculate new iterate implicitly as b+p/q where we arrange
       p >= 0.  The implicit form is used to prevent overflow.
*/
       p = (*b-a)*fb;
       q = fa-fb;
       if (p < zero)
       {
          p = -p;
          q = -q;
       }
/*
       Update a; check if reduction in the size of bracketing
       interval is being reduced at a reasonable rate.  If not,
       bisect until it is.
*/
       a = *b;
       fa = fb;
       count += 1;
       bisect = false;
       if (count >= 4)
       {
          if (eight*acmb >= width)
             bisect = true;
          else
          {
             count = 0;
             width = acmb;
          }
       }
/*
       Test for too small a change.
*/
       if (p <= abs_d(q)*tol)
/*
          If the change is too small, increment by tol.
*/
            *b += sign(tol, cmb);
       else
       {
/*
          Root ought to be between b and (c+b)/2.
*/
          if (p < cmb*q)
             *b += p/q;        /*    Use secant rule.   */
          else
             bisect = true;    /*    Use bisection.     */
       }
       if (bisect)
          *b = *c - cmb;
/*
       The computation for new iterate b has been completed.
*/
       fb = (*f)(*b);
       num_func_evals += 1;
       if (abs_d(fb) == zero)
       {
          *c = *b;
          *residual = fb;
          return 0;
       }
       if (sign(one, fb) == sign(one, fc))
       {
          *c = a;
          fc = fa;
       }
    }
}

